/*
   App to poll an APC UPS (via apcupsd)
*/

'use strict';
var pollint = 10000; // poll every 10 seconds, only changed values will be published

var exec = require('child_process').exec;
var curvalues = {}; // holds current values
var onUpdate = null;

/*
acpi -V
Battery 0: Full, 100%
Battery 0: design capacity 5675 mAh, last full capacity 5584 mAh = 98%
Adapter 0: on-line
Thermal 0: ok, 86.0 degrees C
Thermal 0: trip point 0 switches to mode critical at temperature 115.0 degrees C



apcaccess
APC      : 001,036,0890
DATE     : 2019-09-13 11:19:13 -0400  
HOSTNAME : ryzen3-c70c-samuel
VERSION  : 3.14.12 (29 March 2014) debian
UPSNAME  : CallCenter_1
CABLE    : USB Cable
DRIVER   : USB UPS Driver
UPSMODE  : Stand Alone
STARTTIME: 2019-09-13 11:14:46 -0400  
MODEL    : Back-UPS XS 1500G 
STATUS   : ONLINE 
LINEV    : 121.0 Volts
LOADPCT  : 49.0 Percent
BCHARGE  : 100.0 Percent
TIMELEFT : 12.1 Minutes
MBATTCHG : 5 Percent
MINTIMEL : 3 Minutes
MAXTIME  : 0 Seconds
SENSE    : Medium
LOTRANS  : 88.0 Volts
HITRANS  : 139.0 Volts
ALARMDEL : No alarm
BATTV    : 27.0 Volts
LASTXFER : Unacceptable line voltage changes
NUMXFERS : 0
TONBATT  : 0 Seconds
CUMONBATT: 0 Seconds
XOFFBATT : N/A
SELFTEST : NO
STATFLAG : 0x05000008
SERIALNO : 3B1713X25810  
BATTDATE : 2017-04-02
NOMINV   : 120 Volts
NOMBATTV : 24.0 Volts
NOMPOWER : 865 Watts
FIRMWARE : 866.L9 .D USB FW:L9
END APC  : 2019-09-13 11:19:15 -0400

###########

(cyberpower)
powerstat -status

The UPS information shows as following:

	Properties:
		Model Name................... Back-UPS XS 1500G FW:866.L9 .D USB FW:L9 
		Firmware Number.............. 3B1713X25818  
		Rating Voltage............... 120 V

	Current UPS status:
		State........................ Normal
		Power Supply by.............. Utility Power
		Utility Voltage.............. 121 V
		Battery Capacity............. 100 %
		Remaining Runtime............ 32 min.
		Last Power Event............. None



*/
var wanted = ['upsname', 'serialno', 'battdate', 'model', 'status', 'linev', 'linefreq', 'loadpct', 'battv', 'bcharge', 'timeleft', 'nompower', 'lastxfer'];
function executeCmd(cmd, callback) {

	exec(cmd, function (err, stdout, stderror) {
		// console.log('stdout: %s', output);
		if (err) {
			callback(err);
		}
		else if (stderror) {
			callback(stderror);
		}
		else {
			if (stdout) {
				callback(null, stdout);
			}
			else {
				callback(null, null);
			}
		}
	});
}
var pollers = {
	useProg: '',
	find: function () {

		pollers.apc(function (apc_res) {
			if (apc_res.result == 'error') {

				pollers.cyberpower(function (cyb_res) {
					if (cyb_res.result == 'error') {
						console.error('Could not find any UPS monitoring tool');
						return;
					}
					pollers.useProg = 'cyberpower';
					pollers.poll();
				});
				return;
			}
			pollers.useProg = 'apc';
			pollers.poll();
		})
	},
	poll: function () {
		pollers[pollers.useProg](function (res) {
			if (res.changed && onUpdate) onUpdate(null, curvalues);
			setTimeout(pollers.poll, res.error ? 1000 * 60 : pollint);
		});
	},
	apc: function (cback) {
		executeCmd('apcaccess', function (err, response) {
			if (err) {
				if (err.message.indexOf('Command failed') == -1) {
					console.error(err.message);
					cback({
						result: 'error',
						code: 'unknown',
						message: err.message
					});
					return;
				}
				cback({
					result: 'error',
					code: 'failed',
					message: err.message
				});
				return;
			}
			let changed = false;
			// console.log(response);
			var lines = response.trim().split("\n");

			// loop over every line
			lines.forEach(function (line) {
				// assign values
				var stats = line.split(' : ');
				var label = stats[0].toLowerCase();
				var value = stats[1];

				// remove surrounding spaces
				label = label.replace(/(^\s+|\s+$)/g, '');
				// if found as wanted value, store it
				if (wanted.indexOf(label) > -1) {
					value = value.replace(/(^\s+|\s+$)/g, '');
					// check if value is known, if not store and publish value
					if (curvalues[label] != value) {
						curvalues[label] = value.replace(' Minutes', '').replace(' Volts', '').replace(' Percent', '').replace(' Watts', '');
						changed = true;
					}
				}
			});
			curvalues.usage = parseInt(curvalues.nompower * curvalues.loadpct / 100);
			cback({
				result: 'success',
				changed: changed,
				values: curvalues
			});
		})
	},
	cyberpower: function (cback) {
		executeCmd('pwrstat -status', function (err, response) {
			if (err) {
				if (err.message.indexOf('Command failed') == -1) {
					console.error(err.message);
					cback({
						result: 'error',
						code: 'unknown',
						message: err.message
					});
					return;
				}
				cback({
					result: 'error',
					code: 'failed',
					message: err.message
				});
				return;
			}
			let changed = false;
			// console.log(response);
			var lines = response.trim().split("\n");

			// loop over every line
			lines.forEach(function (line) {
				if (line.indexOf('....') != -1) {
					line = line.trim();
					let val = line.match(/^([a-z\-_\s]+)\.* ([a-z0-9\-_\.\:\s]+)$/i);
					val[1] = val[1].trim();
					val[2] = val[2].trim();
					let field = '', value = '';
					switch (val[1]) {
						case 'Firmware Number':
							field = 'serialno';
							value = val[2];
							break;
						case 'Model Name':
							field = 'model';
							value = val[2].split(' FW:')[0];
							break;
						case 'State':
							field = 'status';
							value = val[2];
							break;
						case 'Utility Voltage':
							field = 'linev';
							value = val[2].replace(/[^0-9]/, '');
							break;
						case 'Battery Capacity':
							field = 'bcharge';
							value = val[2].replace(/[^0-9]/, '');
							break;
						case 'Remaining Runtime':
							field = 'timeleft';
							value = val[2];
							break;
						case 'Last Power Event':
							field = 'lastxfer';
							value = val[2];
							break;
					}
					if (field && value) {
						if (curvalues[field] != value) {
							curvalues[field] = '' + value;
							changed = true;
						}
					}
				}

			});
			cback({
				result: 'success',
				changed: changed,
				curvalues: curvalues
			});
		});
	}
};
pollers.find();
module.exports = function (opts) {
	if (opts && opts.wanted) wanted = opts.wanted;
	if (opts.onUpdate) onUpdate = opts.onUpdate;
	//poll();
	return curvalues;
}