//var ip = require('ip')
var os = require('os'),
	fs = require('fs'),
	net = require('net'),
	util = require('util'),
	spawn = require('child_process').spawn;

var parseLinux = require('./parser/linux')
var parseWin32 = require('./parser/win32')
var parseRow = require('./parser')


const TEN_MEGA_BYTE = 1024 * 1024 * 10
const ONE_MINUTE = 60 * 1000
const options = {
	maxBuffer: TEN_MEGA_BYTE,
	timeout: ONE_MINUTE
}
var servers = [];

function arpScanParse(out) {
	return function lineParser(line) {
		var chunks = line.split('\t');
		out.push({
			ip: chunks[0],
			mac: (chunks[1] || '').toUpperCase(),
			vendor: chunks[2],
			timestamp: Date.now()
		});
	}
}
module.exports.scan = function (iface, cb) {
	var out = [];
	const parser = arpScanParse(out);
	var buffer = '';
	var errstream = '';
	var _doSpawn = function () {
		var arp = spawn("sudo", ["/usr/sbin/arp-scan", '-I', iface, '-l']);
		arp.stdout.on('data', function (data) {
			buffer += data;
		});
		arp.stderr.on('data', function (data) {
			errstream += data;
		});

		arp.on('error', function (err) {
			cb(err, null);
		});
		arp.on('close', function (code) {
			if (code !== 0) {
				console.log("Error running arp-scan " + code + " " + errstream);
				cb(true, code);
				return;
			}
			buffer = buffer.toString().split('\n');
			buffer = buffer.slice(2, -4);
			buffer.forEach(parser);

			cb(false, out);
		});
	}
	if (fs.existsSync('/tmp/arp-scan-result') && fs.statSync('/tmp/arp-scan-result').mtime > new Date().getTime() - 1000 * 60 * 30) {
		fs.readFile('/tmp/arp-scan-result', function (err, data) {
			if (err || !data) {
				_doSpawn();
				return false;
			}
			buffer = data.toString().split('\n');
			buffer = buffer.slice(2, -4);
			buffer.forEach(parser);

			cb(false, out);
		})
		return;
	}
	_doSpawn();

}
/**
 * Read the MAC address from the ARP table.
 * 
 * 3 methods for lin/win/mac  Linux reads /proc/net/arp
 * mac and win read the output of the arp command.
 * 
 * all 3 ping the IP first without checking the response to encourage the
 * OS to update the arp table.
 * 
 * 31/12/2014 -- Changelog by Leandre Gohy (leandre.gohy@hexeo.be)
 * - FIX : ping command for windows (-n not -c)
 *
 * 26/08/2013 -- Changelog by Leandre Gohy (leandre.gohy@hexeo.be)
 * - FIX : arp command for OSX (-n not -an)
 * - MODIFY : rewrite Linux lookup function to avoid looping over all entries and returned lines (arp -n IPADDRESS)
 * - MODIFY : rewrite OSX lookup function to avoid looping over all returned lines
 * - FIX : OSX formates double zero as a single one (i.e : 0:19:99:50:3a:3 instead of 00:19:99:50:3a:3)
 * - FIX : lookup functions did not returns the function on error causing callback to be called twice
 * - FIX : Windows lookup function returns wrong mac address due to indexOf usage (192.168.1.1 -> 192.168.1.10)
 * 
 */
module.exports.getMAC = function (ipaddress, cb) {
	if (process.platform.indexOf('linux') == 0) {
		exports.readMACLinux(ipaddress, cb);
	}
	else if (process.platform.indexOf('win') == 0) {
		exports.readMACWindows(ipaddress, cb);
	}
	else if (process.platform.indexOf('darwin') == 0) {
		exports.readMACMac(ipaddress, cb);
	}
};

module.exports.listALL = function (iface, cb) {
	if (process.platform.indexOf('linux') == 0) {
		exports.listALLLinux(iface, cb);
	}
	else if (process.platform.indexOf('win') == 0) {
		exports.listALLWindows(iface, cb);
	}
	else if (process.platform.indexOf('darwin') == 0) {
		exports.listALLLinux(iface, cb);
	}
};

/**
 * read from arp -a
 */
module.exports.listALLLinux = function (iface, cb) {
	if (typeof iface == 'function') {
		cb = iface;
		iface = '';
	}
	var _args = ['-a'];
	if (iface) {
		_args = ['-i', iface, '-a'];
	}
	var arp = spawn("/usr/sbin/arp", _args);
	var buffer = '';
	var errstream = '';
	arp.stdout.on('data', function (data) {
		buffer += data;
	});
	arp.stderr.on('data', function (data) {
		errstream += data;
	});

	arp.on('close', function (code) {
		if (code !== 0) {
			if (errstream.indexOf("invalid option -- 'i'") != -1) {
				_linuxAlt(iface, cb);
				return;
			}
			console.log("Error running arp " + code + " " + errstream);
			cb(true, code, errstream);
			return;
		}
		var rows = buffer.toString().split('\n');
		cb(false, rows.map(function (row) {
			return row.indexOf('no match found') == -1 ? parseLinux(row, servers) : false;
		}).filter(Boolean));
	});
};
/**
 * read from arp -a
 */
var _linuxAlt = function (iface, cb) {
	if (typeof iface == 'function') {
		cb = iface;
		iface = '';
	}
	var _args = ["-a"];
	if (iface) {
		_args = ['-I', iface, '-a'];
	}
	var arp = spawn("/usr/sbin/arp", _args);
	var buffer = '';
	var errstream = '';
	arp.stdout.on('data', function (data) {
		buffer += data;
	});
	arp.stderr.on('data', function (data) {
		errstream += data;
	});

	arp.on('close', function (code) {
		if (code !== 0) {
			console.log("Error running arp " + code + " " + errstream);
			cb(true, code, errstream);
			return;
		}
		var rows = buffer.toString().split('\n');
		cb(false, rows.map(function (row) {
			return row.indexOf('no match found') == -1 ? parseLinux(row, servers) : false;
		}).filter(Boolean));
	});
};
module.exports.listALLLinuxALT = _linuxAlt;
/**
 * read from arp -n IPADDRESS
 */
module.exports.readMACLinux = function (ipaddress, cb) {

	// ping the ip address to encourage the kernel to populate the arp tables
	var ping = spawn("ping", ["-c", "1", ipaddress]);

	ping.on('close', function (pcode) {
		// not bothered if ping did not work
		//console.log('Ping to ' + ipaddress + ' code ' + pcode)

		var arp = spawn("/usr/sbin/arp", ["-n", ipaddress]);
		var buffer = '';
		var errstream = '';
		arp.stdout.on('data', function (data) {
			buffer += data;
		});
		arp.stderr.on('data', function (data) {
			errstream += data;
		});

		arp.on('close', function (code) {
			if (code !== 0) {
				console.log("Error running arp " + code + " " + errstream);
				cb(true, code);
				return;
			}

			//Parse this format
			//Lookup succeeded : Address                  HWtype  HWaddress           Flags Mask            Iface
			//					IPADDRESS	              ether   MACADDRESS   C                     IFACE
			//Lookup failed : HOST (IPADDRESS) -- no entry
			//There is minimum two lines when lookup is successful
			var table = buffer.toString().trim().split('\n').filter(String);
			if (table.length >= 2) {
				var parts = table[1].split(' ').filter(String);
				cb(false, parts.length == 5 ? parts[2] : parts[1], table);
				return;
			}
			cb(true, "Could not find ip in arp table: " + ipaddress);
		});
	});

};

/**
 * read from arp -a
 */
module.exports.listALLWindows = function (iface, cb) {
	if (typeof iface == 'function') {
		cb = iface;
		iface = '';
	}
	var _args = ["-a"];
	if (iface) {
		_args = ['-I', iface, '-l'];
	}
	var arp = spawn("arp", _args);
	var buffer = '';
	var errstream = '';
	var lineIndex;

	arp.stdout.on('data', function (data) {
		buffer += data;
	});
	arp.stderr.on('data', function (data) {
		errstream += data;
	});

	arp.on('close', function (code) {
		if (code !== 0) {
			console.log("Error running arp " + code + " " + errstream);
			cb(true, code);
			return;
		}
		var rows = buffer.toString().split('\n');
		cb(false, rows.map(function (row) {
			return parseWin32(row, servers)
		}).filter(Boolean));

	});
};

/**
 * read from arp -a IPADDRESS
 */
module.exports.readMACWindows = function (ipaddress, cb) {

	// ping the ip address to encourage the kernel to populate the arp tables
	var ping = spawn("ping", ["-n", "1", ipaddress]);

	ping.on('close', function (pcode) {
		// not bothered if ping did not work
		console.log('Ping to ' + ipaddress + ' code ' + pcode)

		var arp = spawn("arp", ["-a", ipaddress]);
		var buffer = '';
		var errstream = '';
		var lineIndex;

		arp.stdout.on('data', function (data) {
			buffer += data;
		});
		arp.stderr.on('data', function (data) {
			errstream += data;
		});

		arp.on('close', function (code) {
			if (code !== 0) {
				console.log("Error running arp " + code + " " + errstream);
				cb(true, code);
				return;
			}

			var table = buffer.split('\r\n');
			for (lineIndex = 3; lineIndex < table.length; lineIndex++) {
				//parse this format
				//[blankline]
				//Interface: 192.º68.1.54
				//  Internet Address      Physical Address     Type
				//  192.168.1.1           50-67-f0-8c-7a-3f    dynamic

				var parts = table[lineIndex].split(' ').filter(String);
				if (parts[0] === ipaddress) {
					var mac = parts[1].replace(/-/g, ':');
					cb(false, mac);
					return;
				}
			}
			cb(true, "Count not find ip in arp table: " + ipaddress);
		});
	});

};
/**
 * read from arp -n IPADDRESS
 */
module.exports.readMACMac = function (ipaddress, cb) {

	// ping the ip address to encourage the kernel to populate the arp tables
	var ping = spawn("ping", ["-c", "1", ipaddress]);

	ping.on('close', function (code) {
		// not bothered if ping did not work

		var arp = spawn("arp", ["-n", ipaddress]);
		var buffer = '';
		var errstream = '';
		arp.stdout.on('data', function (data) {
			buffer += data;
		});
		arp.stderr.on('data', function (data) {
			errstream += data;
		});

		arp.on('close', function (code) {
			// On lookup failed OSX returns code 1
			// but errstream will be empty
			if (code !== 0 && errstream !== '') {
				console.log("Error running arp " + code + " " + errstream);
				cb(true, code);
				return;
			}

			//parse this format
			//Lookup succeeded : HOST (IPADDRESS) at MACADDRESS on IFACE ifscope [ethernet]
			//Lookup failed : HOST (IPADDRESS) -- no entry
			var parts = buffer.split(' ').filter(String);
			if (parts[3] !== 'no') {
				var mac = parts[3].replace(/^0:/g, '00:').replace(/:0:/g, ':00:').replace(/:0$/g, ':00').replace(/:([^:]{1}):/g, ':0$1:');
				cb(false, mac);
				return;
			}

			cb(true, "Count not find ip in arp table: " + ipaddress);
		});
	});

};